<?php

define('LOG_FILE', realpath(dirname(dirname(__FILE__))) . "/logs/paykeeper.log");

class Utils
{
    const log_file = LOG_FILE;
    public static $serviceInitName = 'paykeeper_payment_system';
    private static $actionFName = 'paykeeper';
    public static $textHiddenForLog = '--hidden-for-log--';
    public static $debug = true;

    /**
     * Logs specified parameters to a log file with a timestamp.
     *
     * This method creates a log entry with the provided parameters and includes a timestamp.
     * If the log file exists and is less than 7 MB in size, the new log entry is prepended to the existing content.
     *
     * @param array $log_params An associative array of parameters to log, where each key is a parameter name and the value is its corresponding data.
     *
     * @return void
     */
    public static function paykeeperLoggerPre($log_params)
    {
        $logText = '--------------START_' . date("Y-m-d_H:i:s") . '--------------' . "\n";
        foreach ($log_params as $key => $value) {
            $logText .= "$key: " . print_r($value,true) . "\n";
        }
        $logText .= "\n\n\n";
        $file = self::log_file;
        if (file_exists($file)) {
            $logSize = filesize($file);
            if ($logSize < (7 * 1024 * 1024)) {
                $logText .= file_get_contents($file);
            }
        }
        file_put_contents($file, $logText);
    }

    /**
     * Logs specified parameters to a log file with a timestamp.
     * If the log file exists and is smaller than 7 MB, the new log entry is prepended to the existing content.
     *
     * @param $logParams
     * @param $isLoggingEnabled
     * @return void
     */
    public static function paykeeperLogger($logParams, $isLoggingEnabled = true)
    {
        if ($isLoggingEnabled) {
            $logText = '--------------START_' . date("Y-m-d_H:i:s") . '--------------' . "\n";
            foreach ($logParams as $key => $value) {
                $logText .= "$key: " . print_r($value, true) . "\n";
            }
            $logText .= "\n\n";

            // Записываем в конец файла с флагом FILE_APPEND
            file_put_contents(self::log_file, $logText, FILE_APPEND);
        }
    }

    /**
     * Types of calculation
     * @param $value
     * @param $type
     * @return string
     */
    public static function getValueTypeParamCart($value, $type)
    {
        return [
            'item_type' => [
                1 => 'goods',
                2 => 'service',
                3 => 'work',
                4 => 'excise',
                5 => 'ip',
                6 => 'payment',
                7 => 'agent',
                16 => 'goods_coded'
            ],
            'payment_type' => [
                1 => 'prepay',
                2 => 'part_prepay',
                3 => 'advance',
                4 => 'full'
            ],
            'measure' => [
                1 => 'pcs',
                2 => 'g',
                3 => 'kg',
                7 => 'm',
                12 => 'l'
            ],
            'vat' => [
                1 => 'none',
                2 => 'vat0',
                7 => 'vat5',
                8 => 'vat7',
                3 => 'vat10',
                4 => 'vat20',
                9 => 'vat105',
                10 => 'vat107',
                5 => 'vat110',
                6 => 'vat120'
            ]
        ][$type][$value];
    }

    /**
     * @return string
     */
    public static function getItemTypeParamCart($value)
    {
        return self::getValueTypeParamCart($value, 'item_type');
    }

    public static function getPaymentTypeParamCart($value)
    {
        return self::getValueTypeParamCart($value, 'payment_type');
    }

    /**
     * @param $tax_rate
     * @param $zero_value_as_none - if variable is set, then when tax_rate is zero, tax is equal to none
     * @param $convert_tax_rate - converts tax rate
     * @return string
     */
    public static function getVat($tax_rate, $zero_value_as_none = true, $convert_tax_rate = false)
    {
        $vat = 'none';
        switch(number_format(floatval($tax_rate), 0, '.', '')) {
            case 0:
                $vat = $zero_value_as_none ? 'none' : 'vat0';
                break;
            case 5:
                $vat = $convert_tax_rate ? 'vat105' : 'vat5';
                break;
            case 7:
                $vat = $convert_tax_rate ? 'vat107' : 'vat7';
                break;
            case 10:
                $vat = $convert_tax_rate ? 'vat110' : 'vat10';
                break;
            case 18:
                $vat = $convert_tax_rate ? 'vat118' : 'vat18';
                break;
            case 20:
                $vat = $convert_tax_rate ? 'vat120' : 'vat20';
                break;
        }
        return $vat;
    }

    /**
     * @return int
     */
    public static function getThisSystemID()
    {
        $paysystem = \Bitrix\Sale\Internals\PaySystemActionTable::getList([
            'order' => [],
            'filter' => [ 'ACTIVE' => 'Y', 'ACTION_FILE' => self::$actionFName ]
        ])->fetch();
        if (!$paysystem) {
            return 0;
        }
        return intval($paysystem['ID']);
    }

    public static function isThisSystemID($paySystemId)
    {
        $service = \Bitrix\Sale\PaySystem\Manager::getObjectById($paySystemId);
        return ($service && $service->getField('ACTION_FILE') === self::$actionFName);
    }

    /**
     * Чистка PAYKEEPER FORM URL от URI для API запросов
     *
     * @param $url
     * @return string
     */
    public static function clearPaykeeperUrl($url)
    {
        foreach (['/create', '/order'] as $uri) {
            if ($pos = strrpos($url, $uri)) {
                $url = substr($url, 0, $pos);
                break;
            }
        }
        return rtrim($url, '/');
    }
}