<?php

// Including the PayKeeper class for payment processing
require_once('paykeeper_common_class/paykeeper.class.php');

/**
 * ControllerExtensionPaymentPaykeeper
 *
 * This controller handles the integration with the PayKeeper payment system.
 * It manages the payment initialization, processing, and response handling
 * from PayKeeper.
 *
 * @package     OpenCart
 * @subpackage  Extension
 * @category    Controller
 * @author      support@paykeeper.ru
 * @license     GPL-3.0-or-later
 */
class ControllerExtensionPaymentPaykeeper extends Controller {
    /**
     * Handles the initial request for the PayKeeper payment extension and loads the payment view.
     *
     * This method loads the necessary language file for the PayKeeper payment extension, prepares
     * the data for the payment confirmation button and the URL for redirecting to the payment page,
     * and then loads the PayKeeper payment template with the relevant data.
     *
     * @return string The rendered PayKeeper payment view template.
     */
    public function index()
    {
        $this->load->language('extension/payment/paykeeper');
        // Передача объекта URL в шаблон
        $data['url_link'] = $this->url->link('extension/payment/paykeeper/gopay', '', true);

        // Ваши другие данные, которые передаются в шаблон
        $data['button_confirm'] = $this->language->get('button_confirm');

        return $this->load->view('extension/payment/paykeeper', $data);
    }

    /**
     * Callback method for handling PayKeeper payment confirmation.
     *
     * This method verifies the integrity of the payment notification data
     * received from PayKeeper. It checks for the correct hash, validates
     * the order existence, and ensures the payment amount matches the order total.
     * Upon successful validation, it updates the order status.
     *
     * @return void Outputs success message or error message based on validation checks.
     */
    public function callback()
    {
        // Exit if no POST data is received
        if (!isset($this->request->post)) {
            exit;
        }

        $this->load->model('checkout/order');

        // Check if required POST parameters are present
        if (!isset($this->request->post['id']) || empty($this->request->post['orderid'])){
            echo "Error! Invalid request!";
            exit;
        }

        // Extract and validate key parameters from POST data
        $secret_seed = $this->config->get('paykeepersecret');
        $id = $this->request->post['id'];
        $sum = $this->request->post['sum'];
        $clientid = $this->request->post['clientid'];
        $orderid = $this->request->post['orderid'];
        $key = $this->request->post['key'];

        // Validate hash to ensure data integrity
        $pk_key = md5($id . sprintf("%.2lf", $sum) . $clientid . $orderid . $secret_seed);
        if (!hash_equals($pk_key, $key)) {
            echo "Error! Hash mismatch";
            exit;
        }

        // Retrieve order details and validate the order ID
        $order_info = $this->model_checkout_order->getOrder($orderid);
        if (!isset($order_info['order_id'])) {
            echo "Error! Order was not found";
            exit;
        }

        // Verify that the order total matches the payment amount
        $order_total = $this->_formatPrice($order_info['total'], $order_info);
        $sum_format = number_format($sum, 2, '.', '');
        if ($order_total !== $sum_format) {
            echo "Error! Incorrect order sum!";
            exit;
        }

        // Update order status to reflect successful payment
        $this->model_checkout_order->addOrderHistory($orderid, $this->config->get('paykeeper_order_status_id'));

        // Output success message with confirmation hash
        echo 'OK ' . md5($id . $secret_seed);
    }

    /**
     * Initiates the payment process with PayKeeper.
     *
     * This method sets up the breadcrumb navigation and page title, generates the payment form,
     * and loads all the necessary page components (e.g., header, footer, columns).
     * If the cart is empty, it redirects the user back to the cart page.
     *
     * @return void Outputs the rendered PayKeeper payment page with an embedded payment form.
     */
    public function gopay()
    {
        $this->load->language('checkout/checkout');
        $this->load->language('extension/payment/paykeeper');

        $data['breadcrumbs'] = array();

        $data['breadcrumbs'][] = array(
            'text' => $this->language->get('text_home'),
            'href' => $this->url->link('common/home')
        );

        $data['breadcrumbs'][] = array(
            'text' => $this->language->get('text_cart'),
            'href' => $this->url->link('checkout/cart')
        );

        $data['breadcrumbs'][] = array(
            'text' => $this->language->get('heading_title'),
            'href' => $this->url->link('checkout/checkout', '', 'SSL')
        );

        $data['breadcrumbs'][] = array(
            'text' => $this->language->get('paykeeper_title'),
            'href' => $this->url->link('extension/payment/paykeeper/gopay', '', 'SSL')
        );

        $data['heading_title'] = $this->language->get('paykeeper_title');
        $this->document->setTitle($data['heading_title']);

        $data['column_left'] = $this->load->controller('common/column_left');
        $data['column_right'] = $this->load->controller('common/column_right');
        $data['content_top'] = $this->load->controller('common/content_top');
        $data['content_bottom'] = $this->load->controller('common/content_bottom');
        $data['footer'] = $this->load->controller('common/footer');
        $data['header'] = $this->load->controller('common/header');

        // Redirect to cart if no products are in the cart
        if (!$this->cart->hasProducts()) {
            $this->response->redirect($this->url->link('checkout/cart', '', true));
        }

        // Generate and assign payment form
        $data["form"] = $this->generatePaymentForm($this->session->data['order_id']);

        // Output the rendered payment page
        $this->response->setOutput($this->load->view('extension/payment/paykeeper_iframe', $data));
    }

    /**
     * @return void
     */
    public function success()
    {
        $this->load->language('checkout/checkout');
        $this->load->language('extension/payment/paykeeper');

        //clear cart
        $this->cart->clear();

        $data['heading_title'] = $this->language->get('paykeeper_title');

        $data['breadcrumbs'] = array();

        $data['breadcrumbs'][] = array(
            'text' => $this->language->get('text_home'),
            'href' => $this->url->link('common/home')
        );

        $data['breadcrumbs'][] = array(
            'text' => $this->language->get('text_cart'),
            'href' => $this->url->link('checkout/cart')
        );

        $data['breadcrumbs'][] = array(
            'text' => $this->language->get('heading_title'),
            'href' => $this->url->link('checkout/checkout', '', 'SSL')
        );

        $data['message'] = $this->language->get('paykeeper_success');

        $this->document->setTitle($data['message']);

        $data['column_left'] = $this->load->controller('common/column_left');
        $data['column_right'] = $this->load->controller('common/column_right');
        $data['content_top'] = $this->load->controller('common/content_top');
        $data['content_bottom'] = $this->load->controller('common/content_bottom');
        $data['footer'] = $this->load->controller('common/footer');
        $data['header'] = $this->load->controller('common/header');

        $this->response->setOutput($this->load->view('extension/payment/paykeeper_feedback', $data));
    }

    /**
     * Displays the success page after a successful PayKeeper payment.
     *
     * This method clears the shopping cart, sets up breadcrumb navigation, and defines the success message.
     * It also loads standard page components (e.g., header, footer) and outputs the feedback page.
     *
     * @return void Renders the payment success feedback page with appropriate data.
     */
    public function failed()
    {
        $this->load->language('checkout/checkout');
        $this->load->language('extension/payment/paykeeper');

        $data['heading_title'] = $this->language->get('paykeeper_title');

        $data['breadcrumbs'] = array();

        $data['breadcrumbs'][] = array(
            'text' => $this->language->get('text_home'),
            'href' => $this->url->link('common/home')
        );

        $data['breadcrumbs'][] = array(
            'text' => $this->language->get('text_cart'),
            'href' => $this->url->link('checkout/cart')
        );

        $data['breadcrumbs'][] = array(
            'text' => $this->language->get('heading_title'),
            'href' => $this->url->link('checkout/checkout', '', 'SSL')
        );

        $data['message'] = $this->language->get('paykeeper_failed');

        $this->document->setTitle($data['message']);

        $data['column_left'] = $this->load->controller('common/column_left');
        $data['column_right'] = $this->load->controller('common/column_right');
        $data['content_top'] = $this->load->controller('common/content_top');
        $data['content_bottom'] = $this->load->controller('common/content_bottom');
        $data['footer'] = $this->load->controller('common/footer');
        $data['header'] = $this->load->controller('common/header');

        $this->response->setOutput($this->load->view('extension/payment/paykeeper_feedback', $data));
    }

    /**
     * Confirms the order by updating its status.
     *
     * This method sets the initial status of the order to "Pending" (status ID 1).
     * It is typically called when a payment has been initiated but not yet completed.
     *
     * @return void Updates the order history with the pending status.
     */
    public function confirm()
    {
        // Load the order model
        $this->load->model('checkout/order');

        // Load the settings model
        $this->load->model('setting/setting');

        // Get the default order status ID from the settings
        $default_order_status_id = $this->config->get('config_order_status_id');

        // Add an order history entry with the default status for the newly created order
        $this->model_checkout_order->addOrderHistory($this->session->data['order_id'], $default_order_status_id);
    }

    /**
     * Get the tax rate for a given sum and tax class.
     *
     * This method retrieves the tax rate associated with a specific amount and tax class.
     * It loops through the tax rates returned by the `getRates` method and returns the first one found.
     * If no tax rate is found, it returns a default rate of 0.
     *
     * @param float $sum The sum for which the tax rate needs to be calculated.
     * @param int $tax_class_id The ID of the tax class to determine the appropriate tax rate.
     *
     * @return int Returns the tax rate as an integer. If no tax rate is found, returns 0.
     */
    protected function getRate($sum, $tax_class_id)
    {
        $tax_data = null;
        foreach($this->tax->getRates($sum, $tax_class_id) as $td) {
            $tax_data = $td;
        }
        return (isset($tax_data['rate'])) ? (int) $tax_data['rate'] : null;
    }

    /**
     * Generates a payment form for PayKeeper payment gateway.
     * This function creates the HTML form for submitting payment details to PayKeeper's payment server
     * and processes additional information such as shipping, low order fees, and cart items.
     *
     * @param int $order_id The unique identifier for the order.
     *
     * @return string HTML content for the payment form or an error message if form generation fails.
     */
    public function generatePaymentForm($order_id)
    {
        $this->load->model('checkout/order');
        $this->load->model('account/order');

        // Instantiate PayKeeper payment object
        $pk_obj = new PaykeeperPayment();

        $order_info = $this->model_checkout_order->getOrder($order_id);

        // Get order totals including low_order_fee, sub_total, shipping, coupon, tax, voucher, total
        $totals_info = $this->model_account_order->getOrderTotals($order_id);
        $loworder_fee_info = [];
        $handling_info = [];
        foreach ($totals_info as $service) {
            if ($service['code'] == 'low_order_fee') {
                $loworder_fee_info = $service;
            } elseif ($service['code'] == 'handling') {
                $handling_info = $service;
            }
        }

        // Check if shipping is free based on coupon
        $shipping_is_free = false;
        if (!empty($this->session->data['coupon'])) {
            $coupon_code = $this->session->data['coupon'];
            $this->load->model('extension/total/coupon');
            $coupon_info = $this->model_extension_total_coupon->getCoupon($coupon_code);
            $shipping_is_free = ($coupon_info['shipping'] == 1);
        }

        // Get shipping details if applicable
        $shipping_info = [];
        $shipping_price = 0;
        if ($this->cart->hasShipping() && !$shipping_is_free) {
            $shipping_info = $this->session->data['shipping_method'];
            $shipping_price = $shipping_info['cost'];
            $shipping_taxes = array("tax" => "none", "tax_sum" => 0);
            $shipping_tax_amount = 0;
            if ($shipping_info["tax_class_id"]) {
                $shipping_tax_rate = $this->getRate($shipping_info['cost'], $shipping_info["tax_class_id"]);
                if (!is_null($shipping_tax_rate)) {
                    $shipping_taxes = $pk_obj->setTaxes($shipping_tax_rate, false);
                    $shipping_tax_amount = $shipping_price * ($shipping_tax_rate / 100);
                }
            }
            $shipping_price = $this->_formatPrice($shipping_price + $shipping_tax_amount, $order_info);
            $shipping_info['cost'] = $shipping_price;
            $shipping_info['tax'] = $shipping_taxes['tax'];
        }

        // Calculate low order fee
        $loworder_price = 0;
        if (!empty($loworder_fee_info)) {
            $loworder_price = $this->config->get('low_order_fee_fee');
            $loworder_taxes = array("tax" => "none", "tax_sum" => 0);
            $loworder_tax_amount = 0;
            if ($this->config->get('low_order_fee_tax_class_id')) {
                $loworder_tax_rate = $this->getRate($loworder_price, $this->config->get('low_order_fee_tax_class_id'));
                if (!is_null($loworder_tax_rate)) {
                    $loworder_taxes = $pk_obj->setTaxes($loworder_tax_rate, false);
                    $loworder_tax_amount = $loworder_price * ($loworder_tax_rate / 100);
                }
            }
            $loworder_price = $this->_formatPrice($loworder_price + $loworder_tax_amount, $order_info);
            $loworder_fee_info['value'] = $loworder_price;
            $loworder_fee_info['tax'] = $loworder_taxes['tax'];
        }

        // Calculate handing
        $handling_price = 0;
        if (!empty($handling_info)) {
            $handling_price = $this->config->get('handling_fee');
            $handling_taxes = array("tax" => "none", "tax_sum" => 0);
            $handling_tax_amount = 0;
            if ($this->config->get('handling_tax_class_id')) {
                $handling_tax_rate = $this->getRate($handling_price, $this->config->get('handling_tax_class_id'));
                if (!is_null($handling_tax_rate)) {
                    $handling_taxes = $pk_obj->setTaxes($handling_tax_rate, false);
                    $handling_tax_amount = $handling_price * ($handling_tax_rate / 100);
                }
            }
            $handling_price = $this->_formatPrice($handling_price + $handling_tax_amount, $order_info);
            $handling_info['value'] = $handling_price;
            $handling_info['tax'] = $handling_taxes['tax'];
        }

        // Calculate the total amount to be paid (excluding low order fee and shipping)
        $paykeeper_total = $order_info['total'] - $loworder_price - $handling_price - $shipping_price;

        // Set order parameters for PayKeeper
        $pk_obj->setOrderParams(
            $this->_formatPrice($paykeeper_total, $order_info), // sum
            trim($order_info['firstname'] . ' ' . $order_info['lastname']),//clientid
            $order_info['order_id'],                            // orderid
            $order_info['email'],                               // client_email
            $order_info['telephone'],                           // client_phone
            '',                                                 // service_name
            $this->config->get('paykeeperserver'),              // payment form url
            $this->config->get('paykeepersecret')               // secret key
        );

        // Generate FZ54 fiscal cart (for products, low order fee, shipping)
        $product_info = $this->cart->getProducts();
        $last_index = 0;
        foreach ($product_info as $item) {
            $taxes = array("tax" => "none", "tax_sum" => 0);
            $tax_amount = 0;
            $name = $item["name"];
            $quantity = floatval($item['quantity']);
            if ($item["tax_class_id"]) {
                $tax_rate = $this->getRate($item['price'], $item["tax_class_id"]);
                if (!is_null($tax_rate)) {
                    $taxes = $pk_obj->setTaxes($tax_rate, false);
                    $tax_amount = $item['price'] * ($tax_rate / 100);
                }
            }
            $price = $this->_formatPrice($item['price'] + $tax_amount, $order_info);


            if ($quantity == 1 && $pk_obj->single_item_index < 0) {
                $pk_obj->single_item_index = $last_index;
            }
            if ($quantity > 1 && $pk_obj->more_then_one_item_index < 0) {
                $pk_obj->more_then_one_item_index = $last_index;
            }

            $pk_obj->updateFiscalCart(
                $pk_obj->getPaymentFormType(),
                $name,
                $price,
                $quantity,
                0,
                $taxes["tax"]
            );

            $last_index++;
        }

        // Set discount information (if applicable)
        $pk_obj->setDiscounts(isset($this->session->data['coupon']) || isset($this->session->data['voucher']));

        // Add low order fee to the fiscal cart
        if ($loworder_price > 0) {
            $pk_obj->updateFiscalCart(
                $pk_obj->getPaymentFormType(),
                $loworder_fee_info['title'],
                $loworder_fee_info['value'],
                1,
                0,
                $loworder_fee_info["tax"]
            );
            $pk_obj->fiscal_cart[$last_index]['item_type'] = 'payment';
            $last_index++;
        }

        // Add low order fee to the fiscal cart
        if ($handling_price > 0) {
            $pk_obj->updateFiscalCart(
                $pk_obj->getPaymentFormType(),
                $handling_info['title'],
                $handling_info['value'],
                1,
                0,
                $handling_info["tax"]
            );
            $pk_obj->fiscal_cart[$last_index]['item_type'] = 'payment';
            $last_index++;
        }

        // Add shipping details to the fiscal cart
        if ($shipping_price > 0) {
            $pk_obj->setShippingPrice($shipping_price);
            if (!$pk_obj->checkDeliveryIncluded($pk_obj->getShippingPrice(), $shipping_info['title'])) {
                $pk_obj->setUseDelivery(); //for precision correct check
                $pk_obj->updateFiscalCart(
                    $pk_obj->getPaymentFormType(),
                    $shipping_info['title'],
                    $shipping_price,
                    1,
                    0,
                    $shipping_info["tax"]
                );
                $pk_obj->delivery_index = $last_index;
                $pk_obj->fiscal_cart[$last_index]['item_type'] = 'service';
            }
        }

        // Set the order total and apply any discounts
        $pk_obj->setOrderTotal($this->_formatPrice($order_info['total'], $order_info));

        // Set discounts
        $pk_obj->setDiscounts($this->config->get("paykeeper_force_discounts_check") == "1");

        // Handle any precision issues
        $pk_obj->correctPrecision();

        return $this->_getPaymentHTMLForm($pk_obj);
    }

    /**
     * Generate the payment form
     *
     * @param $pk_obj
     * @return string
     */
    protected function _getPaymentHTMLForm($pk_obj)
    {
        // Generate the payment form
        $form = "";
        if ($pk_obj->getPaymentFormType() == "create") { //create form
            $to_hash = number_format($pk_obj->getOrderTotal(), 2, ".", "") .
                $pk_obj->getOrderParams("clientid")     .
                $pk_obj->getOrderParams("orderid")      .
                $pk_obj->getOrderParams("service_name") .
                $pk_obj->getOrderParams("client_email") .
                $pk_obj->getOrderParams("client_phone") .
                $pk_obj->getOrderParams("secret_key");
            $sign = hash ('sha256' , $to_hash);

            $form .= '
                <h3>Сейчас Вы будете перенаправлены на страницу банка.</h3>
                <form name="payment" id="pay_form" action="'.$pk_obj->getOrderParams("form_url").'" accept-charset="utf-8" method="post">
                    <input type="hidden" name="sum" value = "'.$pk_obj->getOrderTotal().'"/>
                    <input type="hidden" name="orderid" value = "'.$pk_obj->getOrderParams("orderid").'"/>
                    <input type="hidden" name="clientid" value = "'.$pk_obj->getOrderParams("clientid").'"/>
                    <input type="hidden" name="client_email" value = "'.$pk_obj->getOrderParams("client_email").'"/>
                    <input type="hidden" name="client_phone" value = "'.$pk_obj->getOrderParams("client_phone").'"/>
                    <input type="hidden" name="service_name" value = "'.$pk_obj->getOrderParams("service_name").'"/>
                    <input type="hidden" name="cart" value = \''.htmlentities($pk_obj->getFiscalCartEncoded(),ENT_QUOTES).'\' />
                    <input type="hidden" name="sign" value = "'.$sign.'"/>
                    <input type="submit" id="button-confirm" value="Оплатить"/>
                </form>
                <script text="javascript">
                    window.addEventListener("load", function(){
                        setTimeout(sendForm, 2000);
                    });
                    function sendForm() {
                        $.ajax({ 
                            type: "get",
                            url: "index.php?route=extension/payment/paykeeper/confirm",
                            success: function() {
                                $("#pay_form").submit();
                            }       
                        });
                    }
                    $("#button-confirm").bind("click", sendForm);
                </script>';

        }
        else { //order form
            $payment_parameters = array(
                "clientid"=>$pk_obj->getOrderParams("clientid"),
                "orderid"=>$pk_obj->getOrderParams('orderid'),
                "sum"=>$pk_obj->getOrderTotal(),
                "phone"=>$pk_obj->getOrderParams("phone"),
                "client_email"=>$pk_obj->getOrderParams("client_email"),
                "cart"=>$pk_obj->getFiscalCartEncoded()
            );
            $query = http_build_query($payment_parameters);
            $query_options = array("http"=>array(
                "method"=>"POST",
                "header"=>"Content-type: application/x-www-form-urlencoded",
                "content"=>$query
            ));
            $context = stream_context_create($query_options);

            if( function_exists( "curl_init" )) { //using curl
                $CR = curl_init();
                curl_setopt($CR, CURLOPT_URL, $pk_obj->getOrderParams("form_url"));
                curl_setopt($CR, CURLOPT_POST, 1);
                curl_setopt($CR, CURLOPT_FAILONERROR, true);
                curl_setopt($CR, CURLOPT_POSTFIELDS, $query);
                curl_setopt($CR, CURLOPT_RETURNTRANSFER, 1);
                curl_setopt($CR, CURLOPT_SSL_VERIFYPEER, 0);
                $result = curl_exec( $CR );
                $error = curl_error( $CR );
                if( !empty( $error )) {
                    $form .= "<br/><span class=message>"."INTERNAL ERROR:".$error."</span>";
                }
                else {
                    $form .= $result;
                }
                curl_close($CR);
            }
            else { //using file_get_contents
                if (!ini_get('allow_url_fopen')) {
                    $form .= "<br/><span class=message>"."INTERNAL ERROR: Option allow_url_fopen is not set in php.ini"."</span>";
                }
                else {

                    $form .= file_get_contents($pk_obj->getOrderParams("form_url"), false, $context);
                }
            }
        }
        if ($form == "") {
            $form = '<h3>Произошла ошибка при инциализации платежа</h3>';
        }

        return $form;
    }

    /**
     * Formats the given price according to the order's currency settings.
     *
     * This function takes a price and the order information, formats the price based on
     * the currency code and value of the order, and returns the formatted price with
     * two decimal places.
     *
     * @param float $price The price to be formatted.
     * @param array $order_info An associative array containing order details, including:
     *     - 'currency_code' (string): The currency code (e.g., 'USD', 'EUR').
     *     - 'currency_value' (float): The value of the currency relative to the default currency (usually 1.0).
     *
     * @return string The formatted price with two decimal places.
     *
     * @example
     * $order_info = [
     *     'currency_code' => 'USD',
     *     'currency_value' => 1.0
     * ];
     * $formattedPrice = $this->_formatPrice(100.345, $order_info);
     * echo $formattedPrice; // Outputs: "100.35"
     */
    private function _formatPrice($price, $order_info) {
        $price = $this->currency->format(
            floatval($price),
            $order_info['currency_code'],
            $order_info['currency_value'],
            false
        );
        return number_format($price, 2, '.', '');
    }
}